package gov.va.med.mhv.sm.admin.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.Consumes;
import javax.ws.rs.QueryParam;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import gov.va.med.mhv.sm.admin.data.enums.RelationTypeEnum;
import gov.va.med.mhv.sm.admin.data.enums.SMErrorEnum;
import gov.va.med.mhv.sm.admin.jpa.model.TriageRelation;
import gov.va.med.mhv.sm.admin.jpa.repository.TriageRelationRepository;
import gov.va.med.mhv.sm.admin.service.util.SMHealthshareDelegate;
import gov.va.med.mhv.sm.admin.converter.TriageRelationConverter;
import gov.va.med.mhv.sm.admin.dto.TriageRelationDTO;
import gov.va.med.mhv.sm.admin.dto.TeamDTO;
import gov.va.med.mhv.sm.admin.service.util.ResponseCodeUtil;
import gov.va.med.mhv.sm.healthshare.wsclient.adminqueries.Team;
import gov.va.med.mhv.sm.admin.exception.SMApiException;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.responses.ApiResponse;

@Path("/oerrs")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
@Service("oerrService")
@Component
public class OERRService { 

    @Autowired
    private TriageRelationRepository triageRelationRepository;
	
    @Autowired
    private SMHealthshareDelegate smHealthShareDelegate;
    
	private static final Log log = LogFactory.getLog(OERRService.class);

	@GET
	@Operation(summary = "Get OERR Teams By Triage Group")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = "application/json", 
	        			array = @ArraySchema(schema = @Schema(implementation = TriageRelation.class)))),
			@ApiResponse(responseCode = "404", description = "Teams are not found",content = @Content(mediaType = "application/json", schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "UnExpected Error Occurred",content = @Content(mediaType = "application/json", schema = @Schema(implementation = SMApiException.class)))
	})			
	public List<TeamDTO> getOERRTeamsByTriageGroup(@QueryParam("triageGroupId") Long triageGroupId,
												   @QueryParam("name") String name,
												   @QueryParam("facilityId") String facilityId) throws SMApiException {
		
		List<TriageRelation> teamList = null;
		List<TeamDTO> teamDTOList = null;
		try 
		{
			
			OERREnum.SearchEnum oerrQueryEnum = validateQueryParams(triageGroupId,facilityId,name);
			switch (oerrQueryEnum) {
			case  GET_OERR_BY_TRIAGE_GROUP_ID:
				teamList = triageRelationRepository.getOERRTeamsByTriageGroup(triageGroupId);
				if(teamList != null && teamList.size()>0) {
					teamDTOList = TriageRelationConverter.convertTeamList(teamList);
				}
				return teamDTOList;
			case  GET_OERR_BY_TRIAGE_GROUP_STATION_AND_NAME:
				teamDTOList = getOERRTeamsFromHSByNameAndStation(name,facilityId);
				return teamDTOList;
			default:
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
			
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR);
		}
	}
	
	private List<TeamDTO> getOERRTeamsFromHSByNameAndStation(String name, String facilityId)  throws SMApiException {
		
		List<Team> teamList = null;
		List<TeamDTO> teamDTOList = null;
		try 
		{	
			teamList = smHealthShareDelegate.getOERRTeamsByName(name,facilityId);
				
			if(teamList == null || teamList.size()==0) {
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else {
				teamDTOList = TriageRelationConverter.convertToTeamList(teamList);
			}
		} catch (Exception oerrExp) {
			if(log.isErrorEnabled()){
				log.error("OERRServiceImpl=>Error Occured on searchOERRTeams");
				oerrExp.printStackTrace();
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR);
			}
		}
		return teamDTOList;
	}
	
	private static class OERREnum{
		public enum SearchEnum{
			GET_OERR_BY_TRIAGE_GROUP_ID,
			GET_OERR_BY_TRIAGE_GROUP_STATION_AND_NAME;
		}
	}
	
		
	private OERREnum.SearchEnum validateQueryParams(Long triageGroupId, String facilityId, String name)  throws SMApiException {
		
		if(triageGroupId != null){
			if(name != null || facilityId != null) {
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else {
				return OERREnum.SearchEnum.GET_OERR_BY_TRIAGE_GROUP_ID;
			}
		}else {
			if(name != null && facilityId != null) {
				return OERREnum.SearchEnum.GET_OERR_BY_TRIAGE_GROUP_STATION_AND_NAME;
			}else {
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		}
		
	
	}
}
	
